/*
 * Decompiled with CFR 0.152.
 */
package org.iq80.leveldb.table;

import com.google.common.base.Preconditions;
import com.google.common.base.Throwables;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.channels.FileChannel;
import org.iq80.leveldb.CompressionType;
import org.iq80.leveldb.Options;
import org.iq80.leveldb.table.BlockBuilder;
import org.iq80.leveldb.table.BlockHandle;
import org.iq80.leveldb.table.BlockTrailer;
import org.iq80.leveldb.table.BytewiseComparator;
import org.iq80.leveldb.table.Footer;
import org.iq80.leveldb.table.UserComparator;
import org.iq80.leveldb.util.PureJavaCrc32C;
import org.iq80.leveldb.util.Slice;
import org.iq80.leveldb.util.Slices;
import org.iq80.leveldb.util.Snappy;
import org.iq80.leveldb.util.Zlib;

public class TableBuilder {
    private final int blockRestartInterval;
    private final int blockSize;
    private final CompressionType compressionType;
    private final FileChannel fileChannel;
    private final BlockBuilder dataBlockBuilder;
    private final BlockBuilder indexBlockBuilder;
    private Slice lastKey;
    private final UserComparator userComparator;
    private long entryCount;
    private boolean closed;
    private boolean pendingIndexEntry;
    private BlockHandle pendingHandle;
    private Slice compressedOutput;
    private long position;

    public TableBuilder(Options options, FileChannel fileChannel, UserComparator userComparator) {
        Preconditions.checkNotNull(options, "options is null");
        Preconditions.checkNotNull(fileChannel, "fileChannel is null");
        try {
            Preconditions.checkState(this.position == fileChannel.position(), "Expected position %s to equal fileChannel.position %s", this.position, fileChannel.position());
        }
        catch (IOException e) {
            throw Throwables.propagate(e);
        }
        this.fileChannel = fileChannel;
        this.userComparator = userComparator;
        this.blockRestartInterval = options.blockRestartInterval();
        this.blockSize = options.blockSize();
        this.compressionType = options.compressionType();
        this.dataBlockBuilder = new BlockBuilder((int)Math.min((double)this.blockSize * 1.1, 2097152.0), this.blockRestartInterval, userComparator);
        int expectedNumberOfBlocks = 1024;
        this.indexBlockBuilder = new BlockBuilder(20 * expectedNumberOfBlocks, 1, userComparator);
        this.lastKey = Slices.EMPTY_SLICE;
    }

    public long getEntryCount() {
        return this.entryCount;
    }

    public long getFileSize() throws IOException {
        return this.position + (long)this.dataBlockBuilder.currentSizeEstimate();
    }

    public void add(Slice key, Slice value) throws IOException {
        Preconditions.checkNotNull(key, "key is null");
        Preconditions.checkNotNull(value, "value is null");
        Preconditions.checkState(!this.closed, "table is finished");
        if (this.entryCount > 0L) assert (this.userComparator.compare(key, this.lastKey) > 0) : "key must be greater than last key";
        if (this.pendingIndexEntry) {
            Preconditions.checkState(this.dataBlockBuilder.isEmpty(), "Internal error: Table has a pending index entry but data block builder is empty");
            Slice shortestSeparator = this.userComparator.findShortestSeparator(this.lastKey, key);
            Slice handleEncoding = BlockHandle.writeBlockHandle(this.pendingHandle);
            this.indexBlockBuilder.add(shortestSeparator, handleEncoding);
            this.pendingIndexEntry = false;
        }
        this.lastKey = key;
        ++this.entryCount;
        this.dataBlockBuilder.add(key, value);
        int estimatedBlockSize = this.dataBlockBuilder.currentSizeEstimate();
        if (estimatedBlockSize >= this.blockSize) {
            this.flush();
        }
    }

    private void flush() throws IOException {
        Preconditions.checkState(!this.closed, "table is finished");
        if (this.dataBlockBuilder.isEmpty()) {
            return;
        }
        Preconditions.checkState(!this.pendingIndexEntry, "Internal error: Table already has a pending index entry to flush");
        this.pendingHandle = this.writeBlock(this.dataBlockBuilder);
        this.pendingIndexEntry = true;
    }

    private BlockHandle writeBlock(BlockBuilder blockBuilder) throws IOException {
        Slice raw;
        Slice blockContents = raw = blockBuilder.finish();
        CompressionType blockCompressionType = CompressionType.NONE;
        if (this.compressionType == CompressionType.ZLIB) {
            this.ensureCompressedOutputCapacity(TableBuilder.maxCompressedLength(raw.length()));
            try {
                int compressedSize = Zlib.compress(raw.getRawArray(), raw.getRawOffset(), raw.length(), this.compressedOutput.getRawArray(), 0);
                if (compressedSize < raw.length() - raw.length() / 8) {
                    blockContents = this.compressedOutput.slice(0, compressedSize);
                    blockCompressionType = CompressionType.ZLIB;
                }
            }
            catch (IOException compressedSize) {}
        } else if (this.compressionType == CompressionType.SNAPPY) {
            this.ensureCompressedOutputCapacity(TableBuilder.maxCompressedLength(raw.length()));
            try {
                int compressedSize = Snappy.compress(raw.getRawArray(), raw.getRawOffset(), raw.length(), this.compressedOutput.getRawArray(), 0);
                if (compressedSize < raw.length() - raw.length() / 8) {
                    blockContents = this.compressedOutput.slice(0, compressedSize);
                    blockCompressionType = CompressionType.SNAPPY;
                }
            }
            catch (IOException compressedSize) {
                // empty catch block
            }
        }
        BlockTrailer blockTrailer = new BlockTrailer(blockCompressionType, TableBuilder.crc32c(blockContents, blockCompressionType));
        Slice trailer = BlockTrailer.writeBlockTrailer(blockTrailer);
        BlockHandle blockHandle = new BlockHandle(this.position, blockContents.length());
        this.position += this.fileChannel.write(new ByteBuffer[]{blockContents.toByteBuffer(), trailer.toByteBuffer()});
        blockBuilder.reset();
        return blockHandle;
    }

    private static int maxCompressedLength(int length) {
        return 32 + length + length / 6;
    }

    public void finish() throws IOException {
        Preconditions.checkState(!this.closed, "table is finished");
        this.flush();
        this.closed = true;
        BlockBuilder metaIndexBlockBuilder = new BlockBuilder(256, this.blockRestartInterval, new BytewiseComparator());
        BlockHandle metaindexBlockHandle = this.writeBlock(metaIndexBlockBuilder);
        if (this.pendingIndexEntry) {
            Slice shortSuccessor = this.userComparator.findShortSuccessor(this.lastKey);
            Slice handleEncoding = BlockHandle.writeBlockHandle(this.pendingHandle);
            this.indexBlockBuilder.add(shortSuccessor, handleEncoding);
            this.pendingIndexEntry = false;
        }
        BlockHandle indexBlockHandle = this.writeBlock(this.indexBlockBuilder);
        Footer footer = new Footer(metaindexBlockHandle, indexBlockHandle);
        Slice footerEncoding = Footer.writeFooter(footer);
        this.position += (long)this.fileChannel.write(footerEncoding.toByteBuffer());
    }

    public void abandon() {
        Preconditions.checkState(!this.closed, "table is finished");
        this.closed = true;
    }

    public static int crc32c(Slice data, CompressionType type) {
        PureJavaCrc32C crc32c = new PureJavaCrc32C();
        crc32c.update(data.getRawArray(), data.getRawOffset(), data.length());
        crc32c.update(type.persistentId() & 0xFF);
        return crc32c.getMaskedValue();
    }

    public void ensureCompressedOutputCapacity(int capacity) {
        if (this.compressedOutput != null && this.compressedOutput.length() > capacity) {
            return;
        }
        this.compressedOutput = Slices.allocate(capacity);
    }
}

